/*!---------------------------------------------------------
 * Copyright (C) Microsoft Corporation. All rights reserved.
 *--------------------------------------------------------*/
import { ReportingChannel } from "./reporting-channel";
import { PromiseResults } from "./try-all";
import { EventEmitter } from "events";
/**
 * Constructor configuration options
 *
 * @export
 * @interface ErrorReporterConfig
 */
export interface ErrorReporterConfig {
    /**
     * The absolute path to the root of the app.
     *
     * @type {string}
     */
    appRoot: string;
    /**
     * The name, version, and (optionally) branch of this package. should uniquely identify it.
     *
     * @type {{name: string, version: string, branch?: string}}
     */
    packageInfo: {
        name: string;
        version?: string;
        branch?: string;
    };
    /**
     * The exe name which can override the real name for reporting channels that use it.
     *
     * @type {string}
     */
    exeName?: string;
}
/**
 * An extremely bare bones implementation of an error reporter.
 * Instantiating this object simply allows you to add/remove reporting channels
 * and send error reports using them. It does not have side effects such as handling uncaught exceptions
 * or killing the process once reporting is done. This means it can be used to report non fatal errors safely.
 */
export declare class ErrorReporter {
    /**
     * The internal event emitter. All calls to it should be wrapped in typesafe functions.
     *
     * @protected
     * @type {EventEmitter}
     */
    protected emitter: EventEmitter;
    /**
     * Adds a reporting channel
     *
     * @param {ReportingChannel} channel
     */
    addReportingChannel(channel: ReportingChannel): void;
    /**
     * Removes the given reporting channel, if present
     *
     * @param {ReportingChannel} channel
     */
    removeReportingChannel(channel: ReportingChannel): void;
    /**
     * Creates an instance of ErrorReporter.
     *
     * @param {Options} [options]
     */
    constructor(config: ErrorReporterConfig);
    /**
     * gets the app name passed into the constructor
     *
     * @readonly
     * @type {string}
     */
    appName: string;
    /**
     * gets the app version, if it was passed to the constructor
     *
     * @readonly
     * @type {string}
     */
    appVersion: string;
    /**
     * gets the app root directory as passed into the constructor
     *
     * @readonly
     * @type {string}
     */
    appRoot: string;
    /**
     * gets the app branch as passed into the constructor
     *
     * @readonly
     * @type {string}
     */
    branch: string;
    /**
     * Reports the given error object to all registered reporting channels.
     * Returns a promise which resolves when all of the reporting channels have either succeeded, failed, or timed out.
     */
    reportError(err: Error, errorType: string): Promise<PromiseResults>;
    /**
     * Registers listenrs for internal errors
     *
     * @param {(internalError: InternalErrorDetails) => void} callback
     * - a callback that accepts an object with details about the error, broken into PII and non PII.
     */
    onInternalError(callback: (internalError: InternalErrorDetails) => void): void;
    /**
     * registers listeners to later inform them that it has started reporting
     *
     * @param {(results: PromiseResults) => void} callback the listener callback
     */
    onReportingStarted(callback: () => void): void;
    /**
     * informs registered listeners that error reporting is sstarting
     *
     * @param {PromiseResults} results - the results from each reporting channel
     * @returns {boolean} - whether there were any listeners
     */
    emitReportingStarted(): boolean;
    /**
     * registers listeners to later inform them that it has finished reporting and give them results.
     *
     * @param {(results: PromiseResults) => void} callback the listener callback
     */
    onReportingFinished(callback: (results: PromiseResults) => void): void;
    /**
     * informs registered listeners that error reporting is done and gives them results.
     *
     * @param {PromiseResults} results - the results from each reporting channel
     * @returns {boolean} - whether there were any listeners
     */
    emitReportingFinished(results: PromiseResults): boolean;
}
/**
 * Details about an internal error, passed to the listener.
 *
 * @export
 * @interface InternalErrorDetails
 */
export interface InternalErrorDetails {
    /**
     * Basic properties about the error that occurred which do not contain PII.
     * May or may not be enough to identify the exact problem.
     *
     */
    props: {
        /**
         * The name of the internal error
         *
         * @type {string}
         */
        InternalErrorType: string;
        /**
         * The stack where the internal error occurred, with PII removed
         *
         * @type {string}
         */
        InternalErrorStack: string;
    };
    /**
     * More details about the internal error that occurred, including PII such as full paths.
     *
     */
    PIIProps: {
        /**
         * The message attached to the internal error
         *
         * @type {string}
         */
        InternalErrorMessage: string;
        /**
         * If present, the string it was trying to process when it encountered this error
         *
         * @type {string}
         */
        ProblemString?: string;
        /**
         * The error object it was trying to process when it encountered this error
         *
         * @type {Error}
         */
        OriginalError?: Error;
        /**
         * The raw error object
         *
         * @type {Error}
         */
        RawInternalError: Error;
    };
}
